static char *_desc ="
/*
 *	m3_ident_pcfs.c
 *
 *	ident_fs() module for rmmount(1M) for
 *	detecting FAT12 device
 *
 *	This program should be compiled as shared library
 *	and installed as /usr/lib/fs/pcfs/ident_fs.so.1
 *
 *	The original ident_fs supplied in Solaris identifies
 *	FAT16 as valid, even if mount(1M) won't mount it properly
 *	without \":c\" appended.
 *	This module additionaly checks the FS ID to make sure
 *	it's FAT12. The logic is refined a way further than the original.
 *
 */
";
static char *_rcsid = "$Id: m3_ident_pcfs.c,v 1.5 2000-03-31 18:39:37+09 kabe Exp $";

#include <stdio.h>
#include <unistd.h>	/* lseek, ...*/
#include <strings.h>

#include <rmmount.h>
#include <sys/fs/pc_label.h>

#ifndef FALSE
#define FALSE 0
#endif

char *_ident_fs_string = "pcfs";


/* the exported function */
int
ident_fs(int fd, char *rawpath, int *clean, int verbose)
{
	int ident_fs_offset(int fd, char *rawpath, int *clean, int verbose, off_t offset);
	return ident_fs_offset(fd, rawpath, clean, verbose, 0L);
}

int
ident_fs_offset(int fd, char *rawpath, int *clean, int verbose, off_t offset)
{
	int	i;
	unsigned char	*ptab;
	int	bytes_sect = PC_SECSIZE;	/* 512; default value */
	off_t	pos;
	int	resv_sects;	/* sectors from PBR to FAT */
	unsigned char pbr[512];
	unsigned char fat[512];

	/* Now, seek to the PBR/FAT and check the
	 * first reserved cluster description.
	 */
	pos = offset;
	if ( lseek(fd, pos, SEEK_SET) != pos ) {
		if (verbose) fprintf(stderr, "ident_pcfs: lseek failed to %ld\n", pos);
		return FALSE;
	}
	if ( read(fd, pbr, 512) != 512 ) {
		if (verbose) fprintf(stderr, "ident_pcfs: read bootsector failed on pos=%ld\n", pos);
		return FALSE;
	}
	/* PBR magic */
	/* Some floppys does not have this signature?? */
	if ( pbr[PCFS_SIGN/*0x1fe*/] != 0x55 || pbr[PCFS_SIGN+1] != 0xAA ) {
		if (verbose) fprintf(stderr, "ident_pcfs: no magic in bootsector\n");
		return FALSE;
	}

	if (verbose) {
		long	psects;
		printf("=== ident_pcfs information\n");
		printf("OEM ID: \"%-8.8s\"\n", pbr+0x03);
		printf("bytes/sector: %d\n", ltohs(pbr[PCB_BPSEC]));
		printf("sectors/cluster: %d\n", pbr[PCB_SPC] );
		printf("reserved sectors: %d\n", ltohs(pbr[PCB_RESSEC]));
		printf("FATs: %d\n", pbr[PCB_NFAT]);
		printf("root dirents: %d\n",  ltohs(pbr[PCB_NROOTENT]));
		printf("media type: 0x%02x\n", pbr[PCB_MEDIA]);
		printf("sectors/track: %d\n", ltohs(pbr[PCB_SPT]));
		printf("heads/cylinder: %d\n", ltohs(pbr[PCB_NHEAD]));
		printf("skip from MBR: %d\n", ltohi(pbr[PCB_HIDSEC]));
		psects = ltohs(pbr[PCB_NSEC]);	/* short total sectors */
		if (psects==0) psects = ltohi(pbr[0x20]); /* long sectors */
		printf("sectors in partition: %ld (%ld kbytes)\n",
		    psects, psects*bytes_sect/1024);
		printf("volume serial: %04X-%04X\n",
		    ltohs(pbr[0x27]), ltohs(pbr[0x29]));
		printf("volume label: \"%-11.11s\"\n", pbr + 0x2b);
		printf("filesystem: \"%-8.8s\"\n", pbr + 0x36);
	}

	/* check the jump code at beginning */
	if (pbr[0] != DOS_ID1/*e9*/ && pbr[0] != DOS_ID2a/*eb*/) {
		if (verbose) fprintf(stderr,"ident_pcfs: Initial code in PBR not a jump (0x%02X)\n", pbr[0]);
		return FALSE;
	}

	/* check bytes/sector in PBR  */
	bytes_sect = ltohs(pbr[PCB_BPSEC]);
	if (bytes_sect != 512 && bytes_sect != 1024 && bytes_sect != 2048) {
		if (verbose) fprintf(stderr, "ident_pcfs: sector size in bootsector was %ld\n", bytes_sect);
		return FALSE;
	}

	/* Check the FS ID ("FAT12   ") */
	/* Don't forget floppies; these may not have FS ID, so
	 * skip the check if small-total-sectors is there.
	 */
	if ( ltohs(pbr[PCB_NSEC]) == 0 /*its big*/ &&
	     !!strncmp(pbr + 0x36, "FAT12   ", 8)) {
		if (verbose) fprintf(stderr, "ident_pcfs: FStype of bootsector isn't FAT12\n");
		return FALSE;
	}

	/* Now, read FAT */
	resv_sects = ltohs(pbr[PCB_RESSEC]);
	pos += resv_sects * bytes_sect;
	if ( lseek(fd, pos, SEEK_SET) != pos ) {
		if (verbose) fprintf(stderr, "ident_pcfs: lseek for FAT failed for %ld\n", pos);
		return FALSE;
	}
	if ( read(fd, fat, 512) != 512 ) {
		if (verbose) fprintf(stderr, "ident_pcfs: read FAT failed for %ld\n", pos);
		return FALSE;
	}

	/* the head of the FAT should be F8 FF FF */
	if ( (fat[0] != 0xF8 && fat[0] != pbr[PCB_MEDIA]) || 
	     fat[1] != 0xFF || 
	     fat[2] != 0xFF ) {
		if (verbose) fprintf(stderr, "ident_pcfs: FAT head not F8\n");
		return FALSE;
	}

	*clean = 1;
	return 1;
}

