static char *_desc ="
/*
 *	m3_ident_fdiskfs.c
 *
 *	ident_fs() module for rmmount(1M) for
 *	detecting FDISK partitioned device
 *
 *	This program should be compiled as shared library
 *	and installed in appropriate new directory.
 *
 */
";
static char *_rcsid = "$Id: m3_ident_fdiskfs.c,v 1.6 2000-03-31 18:39:37+09 kabe Exp $";

#include <stdio.h>
#include <unistd.h>	/* lseek, ...*/
#include <strings.h>

#include <rmmount.h>
#include <sys/fs/pc_label.h>

#ifndef FALSE
#define FALSE 0
#endif

char *_ident_fs_string = "fdiskfs";

/* the exported function */
int
ident_fs(int fd, char *rawpath, int *clean, int verbose)
{
	int	i;
	unsigned char	*ptab;		/* partition table in MBR */
	int	ptabnum;		/* partition table index */
	int	bytes_sect = PC_SECSIZE;	/* 512; default value */
	off_t	pos;
	int	resv_sects;	/* sectors from PBR to FAT */
	unsigned char mbr[512];	/* Master Boot Record */
	unsigned char pbr[512];	/* Partition Boot Record */
	unsigned char fat[512];	/* first sector of FAT */

	/* first, read the MBR */
	pos = 0;
	if ( lseek(fd, pos, SEEK_SET) != pos ) {
		perror("ident_fdiskfs: lseek zero");
		return FALSE;
	}
	if ( read(fd, mbr, 512) != 512 ) {
		perror("ident_fdiskfs: read MBR");
		return FALSE;
	}

	/* magic */
	if ( mbr[PCFS_SIGN/*0x1fe*/] != 0x55 || mbr[PCFS_SIGN+1] != 0xAA ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: no magic in MBR\n");
		return FALSE;
	}

	/* scan in the primary partition table */
	for (ptabnum=0; ; ptabnum++) {
		if (ptabnum>=PCFS_NUMPART/*4*/) {
			if (verbose) fprintf(stderr, "ident_fdiskfs: no valid primary partition\n");
			return FALSE;
		}

		ptab = mbr + PCFS_PART/*0x1BE*/ + ptabnum*0x10;

		if (/* boot mark */
		     ptab[0] == 0x80 /*active*/ ||
		     ptab[0] == 0x00 ) {
			if (/* system desc */
			    ptab[4] == DOS_SYSFAT16/*4*/ ||
			    ptab[4] == DOS_SYSHUGE /*6*/ ) {
				/* looks valid */
				break;
			} else
			if (ptab[4] == 0x00) {
				/* empty partition; skip */
				continue;
			} else {
				/* unidentified primary partition signature */
				/* We could check something like
				 * startsect+length<=capacity, validate
				 * the PBR etc, but those needs more access
				 * to media which slows down the ident chain.
				 * Just skip it for now (pretend it's a valid
				 * partition info).
				 */
				 continue;
			}
		}
		/* no boot mark; so it may not be a FDISK */
		if (verbose) fprintf(stderr, "ident_fdiskfs: not a valid partition %d\n", ptabnum+1);
		return FALSE;
	}
	/* ptab points to (valid?) partition table */
	/* Now, validate FAT */

	/* We need at least a bytes/sector for seeking,
	 * but always assuming 512 is not correct.
	 * Unfortunately the way to know this is rather heuristic..
	 */
	
	/* For FDISK disks you cannot assume that MBR
	 * has any geometry data, but if someone left some sane value,
	 * use it.
	 */
	i = ltohs(mbr[PCB_BPSEC/*0x0b*/]);
	if ( (i==512 || i==1024 || i==2048) &&
	    mbr[PCB_NFAT/*0x10*/] == 2 &&
	    ltohs(mbr[PCB_NSEC/*0x13*/]) == 0 && /*bigdisks are zero */
	    (mbr[PCB_MEDIA/*0x15*/] == MD_FIXED/*0xF8*/ ||
	     mbr[PCB_MEDIA] == 0xF0) &&
	    !strncmp(mbr + 0x36, "FAT16   ", 8)
	) {
		/* it looks like an (abandoned?) PBR; so
		 * use the sector size.
		 */
		bytes_sect = i;
		if (verbose) fprintf(stderr,"ident_fdiskfs: using %d as sector size from MBR\n", bytes_sect);
	}
	
	/* Now, seek to the PBR/FAT and check the
	 * first reserved cluster description.
	 */
	/* Use only the LBA addressing (CHS check could be included tho) */
	pos = ltohi(ptab[0x08]) * bytes_sect;
	if ( lseek(fd, pos, SEEK_SET) != pos ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: lseek failed to %ld\n", pos);
		return FALSE;
	}

	if ( read(fd, pbr, 512) != 512 ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: read PBR failed on pos=%ld\n", pos);
		return FALSE;
	}

	/* PBR magic */
	if ( pbr[PCFS_SIGN/*0x1fe*/] != 0x55 || pbr[PCFS_SIGN+1] != 0xAA ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: no magic in PBR\n");
		return FALSE;
	}

	if (verbose) {
		long	psects;
		printf("=== ident_fdiskfs information\n");
		printf("Partition: %d\n", ptabnum+1);
		printf("OEM ID: \"%-8.8s\"\n", pbr+0x03);
		printf("bytes/sector: %d\n", ltohs(pbr[PCB_BPSEC]));
		printf("sectors/cluster: %d\n", pbr[PCB_SPC] );
		printf("reserved sectors: %d\n", ltohs(pbr[PCB_RESSEC]));
		printf("FATs: %d\n", pbr[PCB_NFAT]);
		printf("root dirents: %d\n",  ltohs(pbr[PCB_NROOTENT]));
		printf("media type: 0x%02x\n", pbr[PCB_MEDIA]);
		printf("sectors/track: %d\n", ltohs(pbr[PCB_SPT]));
		printf("heads/cylinder: %d\n", ltohs(pbr[PCB_NHEAD]));
		printf("skip from MBR: %d\n", ltohi(pbr[PCB_HIDSEC]));
		psects = ltohs(pbr[PCB_NSEC]);	/* short total sectors */
		if (psects==0) psects = ltohi(pbr[0x20]); /* long sectors */
		printf("sectors in partition: %ld (%ld kbytes)\n",
		    psects, psects*bytes_sect/1024);
		printf("volume serial: %04X-%04X\n",
		    ltohs(pbr[0x27]), ltohs(pbr[0x29]));
		printf("volume label: \"%-11.11s\"\n", pbr + 0x2b);
		printf("filesystem: \"%-8.8s\"\n", pbr + 0x36);
	}

	/* check the jump code at beginning */
	if (pbr[0] != DOS_ID1/*e9*/ && pbr[0] != DOS_ID2a/*eb*/) {
		if (verbose) fprintf(stderr,"ident_fdiskfs: Initial code in PBR not a jump (0x%02X)\n", pbr[0]);
		return FALSE;
	}

	/* check bytes/sector in PBR (likely to succeed tho) */
	if ( ltohs(pbr[PCB_BPSEC]) != bytes_sect ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: sector size in PBR doesn't agree\n");
		return FALSE;
	}

	/* check the FS ID ("FAT16   ") */
	if ( !!strncmp(pbr + 0x36, "FAT16   ", 8)) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: FStype of PBR isn't FAT16\n");
		return FALSE;
	}

	/* Now, read FAT */
	resv_sects = ltohs(pbr[PCB_RESSEC]);
	pos += resv_sects * bytes_sect;
	if ( lseek(fd, pos, SEEK_SET) != pos ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: lseek for FAT failed for %ld\n", pos);
		return FALSE;
	}
	if ( read(fd, fat, 512) != 512 ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: read FAT failed for %ld\n", pos);
		return FALSE;
	}

	/* the head of the FAT should be F8 FF FF */
	if ( (fat[0] != 0xF8 && fat[0] != pbr[PCB_MEDIA]) || 
	     fat[1] != 0xFF || 
	     fat[2] != 0xFF ) {
		if (verbose) fprintf(stderr, "ident_fdiskfs: FAT head not F8\n");
		return FALSE;
	}

	*clean = 1;
	return 1;
}

