/*
 *	fdiskize.c
 *
 *	Add a fake FDISK partition information to
 *	a superfloppy format
 *
 */ static char *_help=
	"fdiskize [-nkv] [-p partition#] [-o bootfile-output] devicefile";

static char *_rcsid = "$Id: fdiskize.c,v 1.2 2000-02-01 08:00:08+09 kabe Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/fs/pc_label.h>

int
main(int argc, char *argv[])
{
	int	dev;
	unsigned char mbr[PC_SECSIZE];
	int	sectsize;
	int	total_sects;
	int	heads,tracks,sectors;
	int	c,x,i;
	int	opt_p = 1;	/*partition#*/
	char    *opt_o = NULL;	/*output bootsect to*/
	int	opt_n = 0;	/*dryrun*/
	int	opt_k = 0;	/* keep other partition table */
	int	opt_v = 0;	/*dump new bootsect*/

	while ((c = getopt(argc, argv, "nhkvp:o:")) != EOF) {
		switch (c) {
		  case 'p':	/* partition */
			opt_p = atoi(optarg);
			if (opt_p<1 || opt_p>4) {
				fprintf(stderr, "partition should be 1-4\n");
				exit(1);
			}
			break;
		  case 'o':	/* output previous bootsect */
			opt_o = optarg; break;
		  case 'n':	/* dryrun */
			opt_n = 1; break;
		  case 'k':	/* keep other partition table */
			opt_k = 1; break;
		  case 'v':	/*dump new bootsect*/
			opt_v = 1; break;
		  case 'h':
		  case '?':
			fprintf(stderr, "Usage: %s\n", _help);
			exit(1);
		}
	}
	if (optind >= argc) {
		/* no device argument... */
		fprintf(stderr, "Usage: %s\n", _help);
		exit(1);
	}

	dev = open(argv[optind], O_RDWR);
	if (dev<0) {perror(NULL); exit(1);}

	if (lseek(dev, 0, SEEK_SET) != 0) {perror(NULL); exit(1);}
	if (read(dev, mbr, PC_SECSIZE) != 512)	{perror(NULL); exit(1);}

	if (mbr[PCFS_SIGN] != 0x55 || mbr[PCFS_SIGN+1] != 0xaa) {
		fprintf(stderr,"no MBR magic\n"); exit(1);
	}
	if (mbr[0] != DOS_ID1 && mbr[0] != DOS_ID2a) {
		fprintf(stderr,"no MBR bootcode\n"); exit(1);
	}

	sectsize = ltohs(mbr[PCB_BPSEC]);
	switch (sectsize) {
	  case 256: case 512: case 1024: case 2048:
		break;
	  default:
		fprintf(stderr, "sector size=%d, not a DOS filesystem?\n");
		exit(1);
	}

	if (mbr[PCB_MEDIA] != MD_FIXED &&
	    mbr[PCB_MEDIA] != 0xf0 ) {
		fprintf(stderr, "Media descriptor invalid: 0x%02x\n", mbr[PCB_MEDIA]);
		exit(1);
	}

	/* now, get geometry */
	total_sects = ltohs(mbr[PCB_NSEC]);
	if (total_sects == 0) {
		/* bigdisk */
		total_sects = ltohi(mbr[0x20]);
	}
	heads = ltohs(mbr[PCB_NHEAD]);
	sectors = ltohs(mbr[PCB_SPT]);

	if (opt_v) {
		printf("Geometry obtained from MBR:\n");
		printf("	Sector size = %d\n", sectsize);
		printf("	Total sectors = %d (%d kbytes)\n",
		    total_sects, total_sects*sectsize/1024);
		printf("	Heads = %d\n", heads);
		printf("	Sectors = %d\n", sectors);
	}

	/* check for preinstalled partition */
	if ((mbr[PCFS_PART] == 0x80 ||
	    mbr[PCFS_PART] == 0x00 ) &&
	    (mbr[PCFS_PART + 4]== DOS_SYSFAT12 ||
	    mbr[PCFS_PART + 4] == DOS_SYSFAT16 ||
	    mbr[PCFS_PART + 4] == DOS_SYSHUGE ) &&
	    1
	) {
		/* somethings's already there... */
		fprintf(stderr, "Primary partition already exists (start/len = %d/%d)\n",
		    ltohi(mbr[PCFS_PART + 8]), ltohi(mbr[PCFS_PART + 12])
		);
		exit(1);
	}

	/* back it up */
	if (opt_o) {
		FILE	*fp;
		if (opt_o[0] == '-') fp = stdout;
		else	fp = fopen(opt_o, "wb");
		if (!fp) {
			fprintf(stderr, "Can't open %s for writing\n", opt_o);
			exit(1);
		}
		fwrite( mbr, 1, PC_SECSIZE, fp);
		fclose(fp);
	}

	/* prepare it */
	for (i=0; i<PCFS_NUMPART; i++) {
		unsigned char *ptab = mbr + PCFS_PART + i*0x10;
		if (i!=(opt_p-1)) {
			if (!opt_k)
				/*clear the table for other partition*/
				for (x=0;x<0x10;x++) ptab[x]=0;
		} else {
			ptab[0] = 0x00;	/*boot mark*/
			ptab[1] = 0x00;	/*head*/
			ptab[2] = 0x01;	/*sect*/
			ptab[3] = 0x00;	/*cyl*/
			ptab[4] = DOS_SYSHUGE;	/*sysdesc*/
			ptab[5] = ((total_sects-1)/sectors)%heads;	/*head*/
			/*cyl*/ x = (total_sects-1) / sectors / heads;
			ptab[6] = 
				((x>>8)<<6) | (((total_sects-1) % sectors)+1);
			ptab[7] = x & 0xff;
			ptab[8] =ptab[9]=ptab[10]=ptab[11]=0;	/*LBA start*/
			ptab[12] =  total_sects      & 0xff;	/*LBA length*/
			ptab[13] = (total_sects>>8)  & 0xff;
			ptab[14] = (total_sects>>16) & 0xff;
			ptab[15] = (total_sects>>24) & 0xff;
		}
	}

	/* dump it */
	if (opt_v) {
		printf("New boot record:\n");
		for (x=0; x<PC_SECSIZE; x+=16) {
			int	y;
			printf("%04X:", x);
			for (y=0; y<16; y++)
				printf("%c%02X",(y==8)?':':' ', mbr[x+y]);
			printf(" ");
			for (y=0; y<16; y++) {
				c = mbr[x+y];
				putchar( (c<' '||c>0x7f)?'.':c );
			}
			printf("\n");
		}
	}

	/* write it back */
	if (!opt_n) {
		if (lseek(dev, 0, SEEK_SET) != 0) {perror(NULL); exit(1);}
		if (write(dev, mbr, PC_SECSIZE) != 512)	{perror(NULL); exit(1);}
	}

	return 0;
}
